#!/usr/bin/python

# dependencies - visualization of source code dependencies
# Copyright (C) 2009-2022  Joachim Reichel <joachim.reichel@posteo.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

'''Contract strongly connected components to single nodes.

Postprocesses the output of sccmap by contracting strongly connected
components (SCCs) into single nodes. SCCs of size larger than 1 get a
rectangular shape and a label that is the concatenation of the original
labels. Other nodes get a elliptical shape and their original label. The
sccmap command should be called with the -d parameter to not suppress SCCs
of size 1.'''

import re
import sys


def uniq(seq):
    '''Remove duplicates in the given sequence.'''
    result = []
    for i in seq:
        if i not in result:
            result.append(i)
    return result


def node(line, mode, nodes, edges, name):
    '''Process the line in "node" mode.'''
    # end of a node, output contents of "nodes"
    if line == "}":
        mode = "unknown"
        nodes = uniq(nodes)
        if len(nodes) > 1:
            shape = "box"
        else:
            shape = "ellipse"
        label = "\\n".join(sorted(nodes))
        print("  node [label=\"%s\", shape=\"%s\"] \"%s\";" \
            % (label, shape, name))
    # another label for current node, append to "nodes"
    else:
        if line.find("->") != -1:
            matches = re.search(r'(.*) -> (.*)', line)
            nodes.append(matches.group(1).strip("\""))
            nodes.append(matches.group(2).strip("\""))
        else:
            nodes.append(line.strip("\""))
    return (mode, nodes, edges, name)


def edge(line, mode, nodes, edges, name):
    '''Process the line in "edge" mode.'''
    # end of edges, output content of "edges"
    if line == "}":
        mode = "unknown"
        edges = uniq(edges)
        print("  %s" % "\n  ".join(sorted(edges)))
    # another edge, append to "edges"
    else:
        if line.find("->") != -1:
            edges.append(line)
    return (mode, nodes, edges, name)


def switch(line, mode, nodes, edges, name):
    '''Switch to the mode corresponding to line.'''
    # begin of a node
    if line.startswith("digraph cluster_"):
        mode = "node"
        nodes = []
        name = line[len("digraph "):-1].strip()
    # begin of edges
    elif line.startswith("digraph scc_map"):
        mode = "edges"
        edges = []
    # error
    else:
        print("error: cannot handle line \"%s\"" % line, file=sys.stderr)
    return (mode, nodes, edges, name)


def main():
    '''Contract strongly connected components to single nodes.'''
    mode = "unknown"
    name = ""
    nodes = []
    edges = []

    print("digraph foo {")

    for line in sys.stdin:
        line = line.strip().rstrip(";")
        if mode == "node":
            (mode, nodes, edges, name) = node(line, mode, nodes, edges, name)
        elif mode == "edges":
            (mode, nodes, edges, name) = edge(line, mode, nodes, edges, name)
        else:
            (mode, nodes, edges, name) = switch(line, mode, nodes, edges, name)

    print("}")


if __name__ == "__main__":
    main()
